<?php

namespace App\DataTables;

use App\Models\Report;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class ReportDataTable extends DataTable
{
    
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('action', function ($row) {
                return view('datatable.action', compact(['row']))->render();
            })
            ->addColumn('type', function ($report) {
                return $report->type == 0 ? 'Published' : 'Upcoming';
            })
            ->addColumn('reportCategory', function ($report) {
                return $report->reportCategory->category_title;
            })
            ->addColumn('reportSubCategory', function ($report) {
                return $report->reportSubCategory->subcategory_title;
            })
            ->editColumn('p_date', function ($report) {
                return $report->formatted_p_date;
            })
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Report $model): QueryBuilder
    {
        $query = $model->newQuery();
        $query->when(request()->has('type'), function ($query) {
            $query->where('type', request('type'));
        })
        ->when(request()->has('category_id'), function ($query) {
            $query->where('category_id', request('category_id'));
        })
        ->when(request()->has('subcategory_id'), function ($query) {
            $query->where('subcategory_id', request('subcategory_id'));
        });
        return $query->latest();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('report-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            //->dom('Bfrtip')
            ->orderBy(1)
            ->selectStyleSingle()
            ->buttons([
                Button::make('excel'),
                Button::make('csv'),
                Button::make('pdf'),
                Button::make('print'),
                Button::make('reset'),
                Button::make('reload')
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('report_title'),
            Column::make('type'),
            Column::make('reportCategory')
                ->title('Category'),
            Column::make('reportSubCategory')
                ->title('Sub Category'),
            Column::make('no_of_table')
                ->title('Tables'),
            Column::make('pages'),
            Column::make('figure'),
            Column::make('p_date')
            ->title('Published Date'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Report_' . date('YmdHis');
    }
}
