<?php

namespace App\DataTables;

use App\Models\Enquiry;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class EnquiryDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('action', function ($row) {
                return view('datatable.deleteaction', compact(['row']))->render();
            })
            ->addColumn('reffer', function ($row) {
                return "<a href='$row->reffer' class='btn btn-info' target='__blank'>$row->reffer</a>";
            })
            ->addColumn('report_id', function ($report) {
                return $report->report->report_title;
            })
            ->addColumn('created_at', function ($row) {
                $created_at = "";
                if ($row->created_at) {
                    $created_at = date('d-M-Y', strtotime($row->created_at));
                }
                return $created_at;
            })
            ->rawColumns(['reffer', 'action'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Enquiry $model): QueryBuilder
    {
        $query = $model->newQuery();
        $query->when(request()->has('type'), function ($query) {
            $query->where('type', request('type'));
        })
            ->when(request()->has('country_code'), function ($query) {
                $country_code = request('country_code');
                $query->where('country_code','LIKE',"%{$country_code}%");
            })
            ->when(request()->has('report_id'), function ($query) {
                $query->where('report_id', request('report_id'));
            })
            ->when(request()->has('category_id'), function ($query) {
                $categoryId = request('category_id');
                $query->whereHas('report', function ($subQuery) use ($categoryId) {
                    $subQuery->whereHas('reportCategory', function ($subSubQuery) use ($categoryId) {
                        $subSubQuery->where('id', $categoryId);
                    });
                });
            })
            ->when(request()->has('subcategory_id'), function ($query) {
                $categoryId = request('subcategory_id');
                $query->whereHas('report', function ($subQuery) use ($categoryId) {
                    $subQuery->whereHas('reportSubCategory', function ($subSubQuery) use ($categoryId) {
                        $subSubQuery->where('id', $categoryId);
                    });
                });
            })
            ->when(request()->has('start_date'), function ($query) {
                if (request('start_date') != '') {
                    $startDate = request('start_date');
                    $query->where('created_at', '>', $startDate);
                }
            })
            ->when(request()->has('end_date'), function ($query) {
                if (request('end_date') != '') {
                    $query->where('created_at', '>', request('end_date'));
                }
            });
        return $query->latest();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('enquiry-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            //->dom('Bfrtip')
            ->orderBy(1)
            ->selectStyleSingle()
            ->buttons([
                Button::make('excel'),
                Button::make('csv'),
                Button::make('pdf'),
                Button::make('print'),
                Button::make('reset'),
                Button::make('reload')
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('report_id')
                ->title('Report Title'),
            Column::make('name'),
            Column::make('email'),
            Column::make('phone'),
            Column::make('message'),
            Column::make('type'),
            Column::make('reffer'),
            Column::make('created_at')
                ->title('Date'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Enquiry_' . date('YmdHis');
    }
}
