<?php

namespace App\DataTables;

use App\Models\Blog;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class BlogDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('image_url', function ($row) {
                return "<img src='" . $row->image_url . "' height='50px' width='50px' >";
            })
            ->addColumn('action', function ($row) {
                return view('datatable.action', compact(['row']))->render();
            })
            ->addColumn('category_id', function ($report) {
                return $report->blogCategory->subcategory_title;
            })
            ->rawColumns(['action', 'image_url'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Blog $model): QueryBuilder
    {
        $query = $model->newQuery();
        $query->when(request()->has('subcategory_id'), function ($query) {
            $query->where('category_id', request('subcategory_id'));
        })
        ->when(request()->has('category_id'), function ($query) {
            $categoryId = request('category_id');
            $query->whereHas('blogCategory', function ($subQuery) use ($categoryId) {
                $subQuery->whereHas('reportCategory', function ($subSubQuery) use ($categoryId) {
                    $subSubQuery->where('id', $categoryId);
                });
            });
        });
        return $query->latest();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('blog-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            //->dom('Bfrtip')
            ->orderBy(1)
            ->selectStyleSingle()
            ->buttons([
                Button::make('excel'),
                Button::make('csv'),
                Button::make('pdf'),
                Button::make('print'),
                Button::make('reset'),
                Button::make('reload')
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            // Column::make('id'),
            Column::make('category_id')
                ->title('Sub-Domain'),
            Column::make('blog_title'),
            Column::make('slug'),
            Column::make('image_url'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Blog_' . date('YmdHis');
    }
}
